<?php

namespace App\Http\Controllers\Dashboard\Blog;

use App\Domain\Blog\Datatables\PostDatatable;
use App\Domain\Blog\Models\Post;
use App\Domain\Blog\Models\PostCategory;
use App\Domain\Core\Enums\CorePermissions;
use App\Http\Controllers\Dashboard\DashboardController;
use App\Services\ArticlePublisherService;
use App\Support\Dashboard\Crud\WithDatatable;
use App\Support\Dashboard\Crud\WithDestroy;
use App\Support\Dashboard\Crud\WithForm;
use App\Support\Dashboard\Crud\WithStore;
use App\Support\Dashboard\Crud\WithUpdate;
use Collective\Html\FormFacade;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;
class PostController extends DashboardController
{
    use WithDatatable,  WithForm , WithStore ,WithUpdate , WithDestroy;

    protected string $name = 'Post';

    protected string $path = 'dashboard.blog.posts';

    protected string $datatable = PostDatatable::class;

    protected string $model = Post::class;

    protected array $permissions = [CorePermissions::class, 'posts'];

    protected function rules()
    {
        $isUpdating = (bool) request()->route('post'); // Check if this is an update

        return [
            // Basic fields with localization
            //'title' => 'required|array', // Ensures title is an array for multiple locales
            'title' => 'required|string|max:255', // Each localized title must be a string and not too long

            /*
             * 'slug' => [
                'required',
                'array',
                function ($attribute, $value, $fail) use ($isUpdating) {
                    foreach ($value as $locale => $slug) {
                        // Check for unique slug per locale
                        $query = $this->model::where("slug->$locale", $slug);
                        if ($isUpdating) {
                            $query->where('id', '!=', request()->route('post'));
                        }
                        if ($query->exists()) {
                            $fail(__('validation.unique', ['attribute' => "$attribute ($locale)"]));
                        }
                    }
                },
            ],
             'slug' => [
                'required',
                'string',
                'regex:/^[ء-يa-zA-Z0-9-]+$/u', // Allows Arabic, English letters, numbers, and hyphens
            ],
             */

           // 'content' => 'required|array', // Ensures title is an array for multiple locales
            'content' => 'required|string', // Each localized title must be a string and not too long

            // Foreign keys for relationships
            //'category_id' => 'required|exists:post_categories,id',

            // SEO Fields with localization
           // 'seo_title' => 'required|array',
          //  'seo_title' => 'required|string|max:60', // Limits SEO title to 60 characters
            //'seo_description' => 'required|array',
          //  'seo_description' => 'required|string|max:160', // Limits SEO description to 160 characters
           // 'seo_keywords' => 'sometimes|nullable|array',
           // 'seo_keywords' => 'sometimes|nullable|string',

            //'canonical_url' => 'sometimes|nullable|array', // Ensures title is an array for multiple locales
           // 'canonical_url' => 'sometimes|nullable|url', // Each localized title must be a string and not too long

            // Open Graph fields
            //'og_title' => 'sometimes|nullable|array',
           // 'og_title' => 'sometimes|nullable|string|max:60',
            //'og_description' => 'sometimes|nullable|array',
          //  'og_description' => 'sometimes|nullable|string|max:160',

          //  'og_image' => 'sometimes|nullable|mimes:jpeg,png,jpg|max:4048',
          //  'main_image' => 'sometimes|nullable|mimes:jpeg,png,jpg|max:4048',
           // 'og_type' => 'required|string|in:website,article,video,book,profile',

            // Twitter Card fields
            //'twitter_title' => 'sometimes|nullable|array',
          //  'twitter_title' => 'sometimes|nullable|string|max:60',
           // 'twitter_description' => 'sometimes|nullable|array',
          //  'twitter_description' => 'sometimes|nullable|string|max:160',
           // 'twitter_card' => 'required|string|in:summary,summary_large_image,app,player',
           // 'twitter_image' => 'sometimes|nullable|mimes:jpeg,png,jpg|max:4048',

            // Publication controls
           // 'is_published' => 'required|string',
            //'published_at' => 'sometimes|nullable|date', // Optional publish date
        ];
    }
    public function show($id)
    {
        $model = ($this->model)::findOrFail($id);

        $model && FormFacade::setModel($model);
        $data['model'] = $model;
        $data['route'] = $this->path;
        $data['formName'] = $this->name;
        $data['formBreadCrumbs'] = Str::of($this->path)
            ->explode('.')
            ->map(fn ($i) => __(Str::studly($i)))
            ->push(__($model ? 'Edit' : 'Create'));

        return view("{$this->viewPath()}.show", array_merge($this->formData($model), $data));
    }
    public function publish($postId)
    {
        $process = new Process(['php', 'artisan', 'publish:translated-article', $postId]);
        $process->setWorkingDirectory(base_path());
        $process->setTimeout(3600); // 1 hour timeout
        $process->start();

        return response()->stream(function () use ($process) {
            $buffer = '';  // Keep track of what we've output

            while ($process->isRunning()) {
                if ($output = $process->getIncrementalOutput()) {
                    // Add to buffer and send only new content
                    $buffer .= $output;
                    ob_start();
                    echo "data: " . json_encode(['output' => $output]) . "\n\n";
                    ob_end_flush();
                    flush();
                }

                if ($error = $process->getIncrementalErrorOutput()) {
                    ob_start();
                    echo "data: " . json_encode(['error' => $error]) . "\n\n";
                    ob_end_flush();
                    flush();
                }
                usleep(100000);
            }

            // Only send remaining output that wasn't sent incrementally
            $finalOutput = $process->getOutput();
            if ($finalOutput && $finalOutput !== $buffer) {
                $remainingOutput = substr($finalOutput, strlen($buffer));
                if ($remainingOutput) {
                    ob_start();
                    echo "data: " . json_encode(['output' => $remainingOutput]) . "\n\n";
                    ob_end_flush();
                    flush();
                }
            }

            if ($error = $process->getErrorOutput()) {
                ob_start();
                echo "data: " . json_encode(['error' => $error]) . "\n\n";
                ob_end_flush();
                flush();
            }

            ob_start();
            echo "data: " . json_encode(['finished' => true]) . "\n\n";
            ob_end_flush();
            flush();
        }, 200, [
            'Cache-Control' => 'no-cache',
            'Content-Type' => 'text/event-stream',
            'X-Accel-Buffering' => 'no',
        ]);
    }
    protected function formData(?Model $model = null): array
    {
        return [
            //'categories' => toMap(PostCategory::get()),
            'og_types' => [
                'website' => 'Website',
                'article' => 'Article',
                'book' => 'Book',
                'profile' => 'Profile',
                'video.movie' => 'Video (Movie)',
                'video.episode' => 'Video (Episode)',
                'video.tv_show' => 'Video (TV Show)',
                'video.other' => 'Video (Other)',
                'music.song' => 'Music (Song)',
                'music.album' => 'Music (Album)',
                'music.playlist' => 'Music (Playlist)',
                'music.radio_station' => 'Music (Radio Station)',
            ],
            'twitter_cards' => [
                'summary' => 'Summary',
                'summary_large_image' => 'Summary Large Image',
                'app' => 'App',
                'player' => 'Player',
            ],
        ];
    }

    protected function storeAction(array $validated)
    {
        $og_image = Arr::pull($validated, 'og_image');
        $twitter_image = Arr::pull($validated, 'twitter_image');
        $main_image = Arr::pull($validated, 'main_image');
        $validated['user_id'] = auth()->id();
        $model = ($this->model)::create($validated);
        $og_image && $model->addMedia($og_image)->toMediaCollection('og_image');
        $twitter_image && $model->addMedia($twitter_image)->toMediaCollection('twitter_image');
        $main_image && $model->addMedia($main_image)->toMediaCollection('main_image');
    }

    protected function updateAction(array $validated, Model $model)
    {
        $og_image = Arr::pull($validated, 'og_image');
        $main_image = Arr::pull($validated, 'main_image');
        $twitter_image = Arr::pull($validated, 'twitter_image');

        $model->update($validated);
        if (request('og_image_remove')) {
            $model->clearMediaCollection('og_image');
        }
        if (request('twitter_image_remove')) {
            $model->clearMediaCollection('twitter_image');
        }
        if (request('main_image_remove')) {
            $model->clearMediaCollection('main_image');
        }
        if ($main_image instanceof UploadedFile) {
            $model->clearMediaCollection('main_image');
            $model->addMedia($main_image)->toMediaCollection('main_image');
        }
        if ($og_image instanceof UploadedFile) {
            $model->clearMediaCollection('og_image');
            $model->addMedia($og_image)->toMediaCollection('og_image');
        }
        if ($twitter_image instanceof UploadedFile) {
            $model->clearMediaCollection('twitter_image');
            $model->addMedia($twitter_image)->toMediaCollection('twitter_image');
        }
    }
}
