<?php

namespace App\Domain\Core\Datatables\Administration;

use App\Domain\Core\Models\Administration\User;
use App\Support\Dashboard\Datatables\BaseDatatable;
use App\Support\Dashboard\Datatables\Columns\ImageColumn;
use App\Support\Dashboard\Datatables\Columns\UserColumn;
use Illuminate\Database\Eloquent\Builder;
use Yajra\DataTables\Html\Column;

class UserDatatable extends BaseDatatable
{
    protected ?string $actionable = 'edit|delete';
    protected bool $indexer = false;

    public function query(): Builder
    {
        return User::role('user')->with(['company', 'city']);
    }


    protected function columns(): array
    {
        return [
            Column::make('id')->title(__('ID')),
            Column::make('name')->title(__('Name')),
            Column::make('phone')->title(__('Phone')),
            Column::make('status')->title(__('Status')),
            Column::make('city.title.' . app()->getLocale())->title(__('City')),
            Column::make('company.title.' . app()->getLocale())->title(__('Company')),
            Column::make('company_image')->title(__('Company Image')),
            Column::make('orders')->title(__('Orders')),
            Column::make('cars')->title(__('User Cars')),

        ];
    }

    protected function orders(): array
    {
        return [
            'name' => fn($i, $k) => $i->orderBy('name', $k),
        ];
    }

    protected function filters(): array
    {
        return [
            'name' => function ($q, $k) {
                $q->where('name', 'like', "%$k%")
                    ->orWhere('last_name', 'LIKE', "%{$k}%")
                    ->orWhere('first_name', 'LIKE', "%{$k}%");
            },
            'city.title.' . app()->getLocale() => fn($i, $k) => $i->whereHas('city', function ($q) use ($k) {
                $q->where('title', 'like', "%$k%");
            }),
            'company.title.' . app()->getLocale() => fn($i, $k) => $i->whereHas('company', function ($q) use ($k) {
                $q->where('title', 'like', "%$k%");
            }),
        ];
    }

    protected function customColumns(): array
    {
        return [
            'name' => UserColumn::render(),
            'company_image' => function ($model) {
                return ImageColumn::render($model?->getFirstMediaUrl('company_image'));
            },
            'orders' => function ($model) {
                return '<a target="_blank" class="btn btn-sm badge badge-info"
                        href="' . route('dashboard.order.orders.index', ['user_id' => $model->id]) . '">
                 ' . __('Orders')
                    . '</a>';
            },
            'cars' => function ($model) {
                return '<a target="_blank" class="btn btn-sm badge badge-danger"
                        href="' . route('dashboard.user.user-cars.index', ['user_id' => $model->id]) . '">
                 ' . __('User Cars')
                    . '</a>';
            },
        ];
    }
}
