<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

class generateJsonFiles extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'generate-json-files 
    {lang? : The subfolder under "views/frontend" (e.g., ar, en)} 
    {--types=* : The types of generate (e.g., titles,descriptions,change-by-command)} 
    {--dry-run : Preview changes without modifying files}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update <title> and <meta> tags in resource files dynamically with GPT-generated text';


    public function handle()
    {
        $lang = $this->argument('lang'); // Get the folder name
        $types = $this->option('types');

        // If no types specified, ask user to select
        if (empty($types)) {
            $types = $this->choice(
                'What types do you want to generate?',
                ['titles', 'descriptions', 'change-by-command'],
                null,
                null,
                true // Allow multiple selections
            );
        }
        if($lang){
            $directory = resource_path('views/frontend/'.$lang); // Load all folders in 'frontend'
        }else{
            $directory = resource_path('views/frontend'); // Load all folders in 'frontend'
        }

        if (!is_dir($directory)) {
            $this->error("Directory not found: {$directory}");
            return 1;
        }

// Collect files from the iterator into an array
        $files = [];
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($directory, \FilesystemIterator::SKIP_DOTS)
        );

        foreach ($iterator as $fileInfo) {
            if (!$fileInfo->isDir()) { // Only process files, skip directories
                $files[] = $fileInfo->getPathname();
            }
        }

// Sort files as per the machine's filesystem order
        natsort($files); // Natural order sorting

        $processedCount = 0;
        $skippedCount = 0;

        foreach ($files as $file) {

            $fileInfo = new \SplFileInfo($file);
            // Process only .php files and exclude specific ones
            if ($fileInfo->getExtension() !== 'php' || $fileInfo->getFilename() === 'thank-you.blade.php') {
                continue;
            }

            $content = @file_get_contents($file);
            if ($content === false) {
                $this->error("Failed to read file: {$file}");
                $skippedCount++;
                continue;
            }

            // Example: Process file content
            $folderName = basename(dirname($file));
            // Process each selected type
            foreach ($types as $type) {
                $this->info("Processing {$type} for file: {$fileInfo->getFilename()}");

                switch (trim($type)) {
                    case 'titles':
                        $this->generateJsonTitleExamples($content, $file, $folderName);
                        break;

                    case 'descriptions':
                        $this->generateJsonDescriptionExamples($content, $file, $folderName);
                        break;

                    case 'change-by-command':
                        $this->generateChangeByCommandContent($content, $file, $folderName);
                        break;

                    default:
                        $this->warn("Unknown type: {$type}");
                        break;
                }
            }


            unset($content); // Free memory after each file
            $processedCount++;

            // Log progress every 5 files
            if ($processedCount % 5 === 0) {
                $this->info("Processed {$processedCount} files...");
                gc_collect_cycles(); // Force garbage collection
            }
        }

        $this->info("Processing complete. {$processedCount} files processed, {$skippedCount} skipped.");
        return 0;

    }


    /**
     * Process and update the <title> tag.
     *
     * @param string $content
     * @param string $file
     * @return string
     */

    private function generateJsonTitleExamples(string $content, string $file,$lang): string
    {

        $bladeFileName = basename($file, '.blade.php');

        // Regex to match <title> tag and capture the content
        $pattern = '/<title>\s*\{\{\s*(.*?)\s*\}\}\s*<\/title>/i';

        if (preg_match($pattern, $content, $matches)) {
            $oldTitle = $matches[1]; // Captures the Blade expression inside {{ ... }}
            $this->info("Old <title> found in {$file}: '{$oldTitle}'");
            // Generate a new title with OpenAI
            $openAIService = new \App\Services\OpenAITranslationService();
            $newTitle = $openAIService->generateJsonTitleExamplesAndSave($oldTitle,50,$lang,$bladeFileName);
            return  $content;
        }

        $this->warn("No <title> tag found in {$file}");
        return $content; // Return the original content if no match
    }
    private function generateJsonDescriptionExamples(string $content, string $file,$lang): string
    {

        $bladeFileName = basename($file, '.blade.php');

        // Regex to match <title> tag and capture the content
        $pattern = '/<meta\s+name="description"\s+content="(.*?)"\s*\/?>/i';

        if (preg_match($pattern, $content, $matches)) {
            $oldDescription = $matches[1]; // Captures the old description text
            $oldDescription = str_replace('{{','',$oldDescription);
            $oldDescription = str_replace('}}','',$oldDescription);

            $this->info("Old <meta description> found in {$file}: '{$oldDescription}'");
            // Generate a new title with OpenAI
            $openAIService = new \App\Services\OpenAITranslationService();
            $newTitle = $openAIService->generateJsonDescriptionExamplesAndSave($oldDescription,25,$lang,$bladeFileName);
            return  $content;
        }

        $this->warn("No <meta description> tag found in {$file}");
        return $content; // Return the original content if no match
    }

    private function generateChangeByCommandContent(string $content, string $file,$lang): string
    {

        $bladeFileName = basename($file, '.blade.php');
        // Regex to match elements with class="change-by-command" and capture their inner content
        $pattern = '/<(\w+)\s+class=["\']change-by-command["\'].*?>(.*?)<\/\1>/is';
        if (preg_match_all($pattern, $content, $matches, PREG_SET_ORDER)) {

            $openAIService = new \App\Services\OpenAITranslationService();

            foreach ($matches as $match) {
                $tag = $match[1];         // The HTML tag (e.g., p, h2)
                $oldContent = $match[2]; // The inner content of the tag

                $oldContent = str_replace('{{','',$oldContent);
                $oldContent = str_replace('}}','',$oldContent);
                $this->info("Found <{$tag}> with class 'change-by-command' in {$file}: '{$oldContent}'");
                // Generate a new title with OpenAI
                $openAIService = new \App\Services\OpenAITranslationService();
                $newTitle = $openAIService->generateJsonContentExamplesAndSave($oldContent,25,$lang,$bladeFileName);

                return $content; // Return the modified content
            }
        } else {
            $this->warn("No elements with class 'change-by-command' found in {$file}");
        }

        return $content; // Return the modified content
    }



}
